<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
date_default_timezone_set('UTC');

// 🔧 Load config
$config_path = __DIR__ . '/config.php';
if (!file_exists($config_path)) {
    http_response_code(500);
    exit("❌ config.php not found");
}
$config = require $config_path;
$type = strtolower($config['ServerFiles'] ?? 'louis');

$wallet         = $config['usdt_wallet'] ?? '';
$log_file       = __DIR__ . "/usdt-ipn.log";
$processed_file = __DIR__ . "/usdt-processed.json";
$addresses_file = __DIR__ . "/trc_addresses.json";

if (!$wallet) {
    file_put_contents($log_file, "[" . date("Y-m-d H:i:s") . "] ❌ Wallet not defined\n", FILE_APPEND);
    exit("❌ Wallet not configured");
}

$api_url = "https://api.trongrid.io/v1/accounts/$wallet/transactions/trc20?limit=50";

// ✅ Connect to MSSQL
try {
    $pdo = new PDO(
        "dblib:host={$config['mssql_host']};dbname={$config['mssql_db']}",
        $config['mssql_user'],
        $config['mssql_password']
    );
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    file_put_contents($log_file, "[" . date("Y-m-d H:i:s") . "] ❌ MSSQL ERROR: " . $e->getMessage() . "\n", FILE_APPEND);
    exit;
}

// ✅ Load data
$processed     = file_exists($processed_file) ? json_decode(file_get_contents($processed_file), true) : [];
$trc_addresses = file_exists($addresses_file) ? json_decode(file_get_contents($addresses_file), true) : [];
if (!is_array($trc_addresses)) $trc_addresses = [];

$new_txns = [];

// ✅ Fetch TRC20 transactions
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $api_url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
curl_setopt($ch, CURLOPT_TIMEOUT, 10);
curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0');

$response = curl_exec($ch);
if (curl_errno($ch)) {
    file_put_contents($log_file, "[" . date("Y-m-d H:i:s") . "] ❌ cURL error: " . curl_error($ch) . "\n", FILE_APPEND);
    curl_close($ch);
    exit;
}
curl_close($ch);

$data = json_decode($response, true);
if (!isset($data['data'])) {
    file_put_contents($log_file, "[" . date("Y-m-d H:i:s") . "] ❌ Invalid API response\n", FILE_APPEND);
    exit;
}

// ✅ Transaction processing
foreach ($data['data'] as $txn) {
    if (!($txn['confirmed'] ?? false)) continue;

    $id = $txn['transaction_id'] ?? '';
    $to = $txn['to'] ?? '';
    $symbol = $txn['token_info']['symbol'] ?? '';
    $value = $txn['value'] ?? 0;
    $decimals = $txn['token_info']['decimals'] ?? 6;
    $from = $txn['from'] ?? '';
    $ts = date("Y-m-d H:i:s", ($txn['block_timestamp'] ?? 0) / 1000);

    if (!$id || $symbol !== 'USDT' || $to !== $wallet || in_array($id, $processed)) continue;

    $amount = $value / pow(10, $decimals);

    foreach ($trc_addresses as $key => $entry) {
        if (
            strcasecmp($entry['trc_address'] ?? '', $from) === 0 &&
            $amount >= floatval($entry['amount'] ?? 0)
        ) {
            $charname     = $entry['charname'] ?? '';
            $currency     = $entry['currency'] ?? 'WCoinC';
            $amountToAdd  = floatval($entry['amount']);

            // 🔍 Get AccountID by Character Name
            $stmt = $pdo->prepare("SELECT AccountID FROM Character WHERE Name = ?");
            $stmt->execute([$charname]);
            $account = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($account) {
                $accountID = $account['AccountID'];

                // 🔁 Adjust target field name and table
                $field = match ($currency) {
                    'WCoinP'      => 'WCoinP',
                    'GoblinPoint' => 'GoblinPoint',
                    default       => 'WCoinC'
                };

                switch ($type) {
                    case 'ignc':
                        $table = 'T_InGameShop_Point';
                        break;
                    case 'ssemu':
                        $table = null; // ❌ нет CashShop таблицы
                        break;
                    default:
                        $table = 'CashShopData';
                        break;
                }

                if (!$table) {
                    file_put_contents($log_file, "[$ts] ⚠️ Skipped — CashShop table not available for $type\n", FILE_APPEND);
                    break;
                }

                // ✅ Update
                $update = $pdo->prepare("UPDATE $table SET $field = ISNULL($field, 0) + ? WHERE AccountID = ?");
                $success = $update->execute([$amountToAdd, $accountID]);

                if ($success) {
                    $log = "[$ts] ✅ $amount USDT from $from → $accountID +$amountToAdd $currency (char: $charname, txid: $id)\n";
                    file_put_contents($log_file, $log, FILE_APPEND);
                    $new_txns[] = $id;
                    unset($trc_addresses[$key]);
                } else {
                    file_put_contents($log_file, "[$ts] ❌ Failed to update $currency for $accountID\n", FILE_APPEND);
                }
            } else {
                file_put_contents($log_file, "[$ts] ❌ Character not found: $charname\n", FILE_APPEND);
            }

            break;
        }
    }
}

// 📦 Save
if (empty($new_txns)) {
    file_put_contents($log_file, "[" . date("Y-m-d H:i:s") . "] ⏳ No new USDT transactions\n", FILE_APPEND);
} else {
    file_put_contents($processed_file, json_encode(array_merge($processed, $new_txns), JSON_PRETTY_PRINT));
    file_put_contents($addresses_file, json_encode(array_values($trc_addresses), JSON_PRETTY_PRINT));
    echo "✅ Processed " . count($new_txns) . " transactions.\n";
}
